/*---------------------------------------------------------------------------*\

    FILE....: TIMP.CPP
    TYPE....: C++ program
    AUTHOR..: David Rowe
    DATE....: 13/6/01

    Test program for sampling the impulse response of a hybrid/line 
    combination.

    Instructions:

    1. Connect a phone line to port 1 (furtheest from motherboard) of the card.
    2. Start this program running: "./timp test.wav"
    3. Dial the number of the line connected to the port with another phone.
    4. A "zipping" sound will be heard in the phone, and the program will exit
       after a few seconds.

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

#include "../src/mess.h"
#include "../src/comm.h"
#include "../src/config.h"
#include "../src/dspfifo.h"
#include "../src/timer.h"
#include "../src/timer.h"
#include "../src/wobbly.h"
#include "../src/wavecpp.h"
#include "../src/vpbapi.h"
#include "../src/mapdev.h"
#include "../src/translate.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <math.h>

int kbhit();

#define WAIT		1	// time out delay 
#define	BOARD		0	// board number
#define	N	        320	// size of processing frame
#define LOG_FRAMES      100     // number of frames to log
#define IMPULSE         10000   // impulse amplitude

word offhook[] = {
	PC_LCODEC_OFFHK,
	PC_CODEC_OFFHK,
	0			// channel
};

word onhook[] = {
	PC_LCODEC_ONHK,
	PC_CODEC_ONHK,
	0			// channel
};

void set_codec_reg(int    chdev,
	           unsigned short addr, 
          	   unsigned short data,  
		   Comm   *c
		  );

int arg_exists(int argc, char *argv[], char *arg);
void play_set_hw_gain(int handle, float gain, Comm *c);
void record_set_hw_gain(int handle, float gain, Comm *c);

int main(int argc, char *argv[])
{
	long		i;
	VPBREG		*v;
	word		upmess[COMM_MAX_MESS];
	char		s[MAX_STR];
	FILE		*fimp;
	int		ch1_offhook;
	short		buf[N], txbuf[N];
	float           av[N];
	int		frames;
	int             arg,ch,bal,bal2,bal3;
	int             e_frames;
	float           e_echo, erl;
	int             dport,hdial,port;
	float           playg, recg;

	if (argc < 2) {
		printf("usage: %s [-oh] [-port] "
		       "[-i impfile.txt] [-dial port number] "
		       "[-bal1 bal1] [-bal2 bal2] [-bal3 bal3]\n"
		       ,argv[0]);
		exit(0);
	}

	for(i=0; i<N; i++) {
		txbuf[i] = 0;
		av[i] = 0.0;
	}
	txbuf[0] = IMPULSE;

	// initialise 

	try {

	        if ((arg = arg_exists(argc, argv, "-port"))) {
			port = atoi(argv[arg+1]);
			ch = port-1;
		}
		else
			ch = 0;
		
		// use high level API to dial out
		// the transition to low level will cause a momentary
		// on-hook but should work OK

		if ((arg = arg_exists(argc, argv, "-dial")) != 0) {
			dport = atoi(argv[arg+1]);
			assert((dport >= 1) && (dport <= 4));
			assert(dport != port);
			hdial = vpb_open(1, dport);
			vpb_sethook_sync(hdial, VPB_OFFHOOK);
			printf("OK, dialing %s down port %d\n", argv[arg+2],
			       dport);
			vpb_dial_sync(hdial, argv[arg+2]);
			vpb_close(hdial);
		}

		Comm comm;
		v = comm.vpbreg(BOARD);

		// configure VPB signal processing --------------------------

		config_create_object(&comm, BOARD, CODEC_AD , 0, ch, 0);
		config_create_object(&comm, BOARD, ALAW2LIN , 1, 0, 0);
		config_create_object(&comm, BOARD, FIFO_UP  , 2, 0, 
				     v->szrxdf[0]);
		config_create_object(&comm, BOARD, FIFO_DOWN, 3, 0, 
				     v->sztxdf[0]);
		config_create_object(&comm, BOARD, LIN2ALAW , 4, 0, 0);
		config_create_object(&comm, BOARD, CODEC_DA , 5, ch, 0);


		config_create_wire(&comm, BOARD, 0, 1);
		config_create_wire(&comm, BOARD, 1, 2);
      		config_create_wire(&comm, BOARD, 3, 4);
      		config_create_wire(&comm, BOARD, 4, 5);

		if ((arg = arg_exists(argc, argv, "-dial")) != 0) {
			// make sure dialling port remains off-hook
			offhook[2] = hdial;
			comm.PutMessageVPB(BOARD,offhook);
		}

		onhook[2] = ch;
		comm.PutMessageVPB(BOARD,onhook);
		ch1_offhook = 0;

		if ((arg = arg_exists(argc, argv, "-bal1")) != 0) {
			bal = strtol(argv[arg+1], NULL, 16);
			assert((bal >= 0) && (bal <= 255));
			set_codec_reg(ch, 0x32, bal, &comm);
		}

		// option to set codec balance reg 2
		bal2 = -1;
		if ((arg = arg_exists(argc, argv, "-bal2")) != 0) {
			bal2 = strtol(argv[arg+1], NULL, 16);
			assert((bal2 >= 0) && (bal2 <= 255));
			set_codec_reg(ch, 0x3a, bal2, &comm);
		}

		// option to set codec balance reg 3
		bal3 = -1;
		if ((arg = arg_exists(argc, argv, "-bal3")) != 0) {
			bal3 = strtol(argv[arg+1], NULL, 16);
			assert((bal3 >= 0) && (bal3 <= 255));
			set_codec_reg(ch, 0x42, bal3, &comm);
		}

		if (arg_exists(argc, argv, "-oh")) {
			offhook[2] = ch;
			comm.PutMessageVPB(BOARD,offhook);
			ch1_offhook = 1;
			printf("testing.....\n");
		}
		else
			printf("Waiting for ring on port %d\n",port);

		// make sure tx & rx gains set
		playg = -1;
		if ((arg = arg_exists(argc, argv, "-pg"))) {
			playg = atof(argv[arg+1]);
			assert((playg >= -12.0) && (playg <= 12.0));
		}
		play_set_hw_gain(ch, playg, &comm);
		printf("play hw gain set to %3.2fdB\n", playg);

		recg = 12;
		if ((arg = arg_exists(argc, argv, "-rg"))) {
			recg = atof(argv[arg+1]);
			assert((recg >= -12.0) && (recg <= 12.0));
		}
		record_set_hw_gain(ch, recg, &comm);
		printf("rec hw gain set to %3.2fdB\n", recg);

		// default set codec diff amp (V4PCI only)
		set_codec_reg(ch, 0x32, 0xc7, &comm);
		set_codec_reg(ch, 0x42, 0x00, &comm);

		// start main processing loop ------------------------------
		
		mess_mprintf_on();
		config_run(&comm, BOARD);
		i = 0;
		frames = 0;
		e_echo = 0.0; e_frames=0;
		do {
			// check for events from VPB

			while(comm.GetMessageVPB(BOARD, upmess) == OK) {
				translate(BOARD, s, upmess);
				mprintf(s);
				if (upmess[1] == DSP_CODEC_RING) {
					offhook[2] = upmess[2];
					comm.PutMessageVPB(BOARD,offhook);
					
					if (upmess[2] == ch) {
						ch1_offhook = 1;
						printf("testing.....\n");
					}
				
				}
			}

			if (ch1_offhook) {

				while(v->rxdf[0]->Read((word*)buf, N) != OK)
					comm.CheckForAssert(BOARD);
				if (frames > 11) {
					for(i=0; i<N; i++) {
						av[i] += buf[i];
					}
					e_frames++;
				}
				
				while(v->txdf[0]->Write((word*)txbuf, N) != OK)
					comm.CheckForAssert(BOARD);
				
				frames++;
			}

		} while(frames < LOG_FRAMES);

		// clean up and finish 

		onhook[2] = ch;
		comm.PutMessageVPB(BOARD,onhook);
		if ((arg = arg_exists(argc, argv, "-dial")) != 0) {
		  onhook[2] = hdial;
		  comm.PutMessageVPB(BOARD,onhook);
		}
		vpb_sleep(300);
		ch1_offhook = 0;
		mess_mprintf_off();

		// send clear config message -------------------------------

		config_stop(&comm, BOARD);
		config_clear(&comm, BOARD);

		if ((arg = arg_exists(argc, argv, "-i"))) {
			fimp = fopen(argv[arg+1],"wt");
			assert(fimp != NULL);
		}
		else
			fimp = NULL;
		for(i=0; i<N; i++) {
			e_echo += pow(av[i]/e_frames,2.0);
			if (fimp != NULL)
				fprintf(fimp,"%f\n",av[i]/e_frames);
		}
		erl = -10.0*log10(pow(IMPULSE,2.0)/e_echo);
		printf("\nerl = %3.2fdB\n", erl);

	}	// try ...

	catch (Wobbly w) {
		char	s[MAX_STR];
		
		w.translate(s);
		
		if (w.file[0] != 0) 
			printf("exception caught: %s, %s, line = %d\n",s, 
			       w.file, w.line);
		else
			printf("exception caught: %s\n",s);

		printf("Press any key to exit....\n");
	}

	return 0;
}

int arg_exists(int argc, char *argv[], char *arg) {
  int i;

  for(i=0; i<argc; i++)
    if (strcmp(argv[i],arg) == 0)
      return i;

  return 0;
}






